/*
 * Copyright (C) 2010 ZXing authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
 * These authors would like to acknowledge the Spanish Ministry of Industry,
 * Tourism and Trade, for the support in the project TSI020301-2008-2
 * "PIRAmIDE: Personalizable Interactions with Resources on AmI-enabled
 * Mobile Dynamic Environments", led by Treelogic
 * ( http://www.treelogic.com/ ):
 *
 *   http://www.piramidepse.com/
 */

package com.google.zxing.client.result;

import java.util.HashMap;
import java.util.Map;

import com.google.zxing.BarcodeFormat;
import com.google.zxing.Result;

/**
 * Parses strings of digits that represent a RSS Extended code.
 * 
 * @author Antonio Manuel Benjumea Conde, Servinform, S.A.
 * @author Agustín Delgado, Servinform, S.A.
 */
public final class ExpandedProductResultParser extends ResultParser {
	enum AI {
		A00("00"),A01("01"),A10("10"),A11("11"),A13("13"),A15("15"),A17("17"),A3100("3100"),A3101("3101"),A3102("3102"),A3103("3103"),A3104("3104"),A3105("3105"),A3106("3106"),A3107("3107"),A3108("3108"),A3109("3109"),A3200("3200"),A3201("3201"),A3202("3202"),A3203("3203"),A3204("3204"),A3205("3205"),A3206("3206"),A3207("3207"),A3208("3208"),A3209("3209"),A3920("3920"),A3921("3921"),A3922("3922"),A3923("3923"),A3930("3930"),A3931("3931"),A3932("3932"),A3933("3933");
		private String value;
		AI(String value){
			this.value = value;
		}
		
		public int length(){
			return this.value.length();
		}
		public String toString(){
			return this.value;
		}
		
		public String substring(int beginIndex){
			return this.value.substring(beginIndex);
		}
		
		public String substring(int beginIndex,int endIndex){
			return this.value.substring(beginIndex,endIndex);
		}
		
	}
	@Override
	public ExpandedProductParsedResult parse(Result result) {
		BarcodeFormat format = result.getBarcodeFormat();
		if (format != BarcodeFormat.RSS_EXPANDED) {
			// ExtendedProductParsedResult NOT created. Not a RSS Expanded
			// barcode
			return null;
		}
		String rawText = getMassagedText(result);

		String productID = null;
		String sscc = null;
		String lotNumber = null;
		String productionDate = null;
		String packagingDate = null;
		String bestBeforeDate = null;
		String expirationDate = null;
		String weight = null;
		String weightType = null;
		String weightIncrement = null;
		String price = null;
		String priceIncrement = null;
		String priceCurrency = null;
		Map<String, String> uncommonAIs = new HashMap();

		int i = 0;

		while (i < rawText.length()) {
			 
			AI ai = AI.valueOf("A"+findAIvalue(i, rawText));
			if (ai == null) {
				// Error. Code doesn't match with RSS expanded pattern
				// ExtendedProductParsedResult NOT created. Not match with RSS
				// Expanded pattern
				return null;
			}
			i += ai.length() + 2;
			String value = findValue(i, rawText);
			i += value.length();

			switch (ai) {
			case A00:
				sscc = value;
				break;
			case A01:
				productID = value;
				break;
			case A10:
				lotNumber = value;
				break;
			case A11:
				productionDate = value;
				break;
			case A13:
				packagingDate = value;
				break;
			case A15:
				bestBeforeDate = value;
				break;
			case A17:
				expirationDate = value;
				break;
			case A3100:
			case A3101:
			case A3102:
			case A3103:
			case A3104:
			case A3105:
			case A3106:
			case A3107:
			case A3108:
			case A3109:
				weight = value;
				weightType = ExpandedProductParsedResult.KILOGRAM;
				weightIncrement = ai.substring(3);
				break;
			case A3200:
			case A3201:
			case A3202:
			case A3203:
			case A3204:
			case A3205:
			case A3206:
			case A3207:
			case A3208:
			case A3209:
				weight = value;
				weightType = ExpandedProductParsedResult.POUND;
				weightIncrement = ai.substring(3);
				break;
			case A3920:
			case A3921:
			case A3922:
			case A3923:
				price = value;
				priceIncrement = ai.substring(3);
				break;
			case A3930:
			case A3931:
			case A3932:
			case A3933:
				if (value.length() < 4) {
					// The value must have more of 3 symbols (3 for currency and
					// 1 at least for the price)
					// ExtendedProductParsedResult NOT created. Not match with
					// RSS Expanded pattern
					return null;
				}
				price = value.substring(3);
				priceCurrency = value.substring(0, 3);
				priceIncrement = ai.substring(3);
				break;
			default:
				// No match with common AIs
				uncommonAIs.put(ai.toString(), value);
				break;
			}
		}

		return new ExpandedProductParsedResult(rawText, productID, sscc,
				lotNumber, productionDate, packagingDate, bestBeforeDate,
				expirationDate, weight, weightType, weightIncrement, price,
				priceIncrement, priceCurrency, uncommonAIs);
	}

	private static String findAIvalue(int i, String rawText) {
		char c = rawText.charAt(i);
		// First character must be a open parenthesis.If not, ERROR
		if (c != '(') {
			return null;
		}

		CharSequence rawTextAux = rawText.substring(i + 1);

		StringBuilder buf = new StringBuilder();
		for (int index = 0; index < rawTextAux.length(); index++) {
			char currentChar = rawTextAux.charAt(index);
			if (currentChar == ')') {
				return buf.toString();
			} else if (currentChar >= '0' && currentChar <= '9') {
				buf.append(currentChar);
			} else {
				return null;
			}
		}
		return buf.toString();
	}

	private static String findValue(int i, String rawText) {
		StringBuilder buf = new StringBuilder();
		String rawTextAux = rawText.substring(i);

		for (int index = 0; index < rawTextAux.length(); index++) {
			char c = rawTextAux.charAt(index);
			if (c == '(') {
				// We look for a new AI. If it doesn't exist (ERROR), we coninue
				// with the iteration
				if (findAIvalue(index, rawTextAux) == null) {
					buf.append('(');
				} else {
					break;
				}
			} else {
				buf.append(c);
			}
		}
		return buf.toString();
	}

}
